<?php
namespace App\Controller;

use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Symfony\Component\Security\Csrf\TokenGenerator\UriSafeTokenGenerator;

use App\Entity\Pais;
use App\Entity\Usuario;
use App\Entity\UsuarioToken;
use App\Entity\Persona;
use App\Entity\Permiso;
use App\Entity\UsuarioPermiso;

use App\Form\Usuario\UsuarioNuevoType;
use App\Form\Usuario\UsuarioReiniciarType;
use App\Form\Usuario\UsuarioReestablecerType;

class UsuarioController extends AbstractController
{
    public function reiniciar(Request $request, \Swift_Mailer $mailer, UserPasswordEncoderInterface $encoder)
    {
        if($this->isGranted(['IS_AUTHENTICATED_REMEMBERED'])) {
            return $this->redirectToRoute('logout');
        }

        $dataForm = ['username' => null, 'email' => null];
        $form = $this->createForm(UsuarioReiniciarType::class, $dataForm);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if($form->isValid()) {
                $data = $form->getData();
                $usuario = $this->getDoctrine()->getRepository(Usuario::class)
                    ->findOneBy([
                        'username' => $data['username'], 
                        'email' => $data['email']
                    ]);

                if($usuario) {
                    $usuarioToken = $this->generarTokenReiniciar($usuario);
                    $this->correoReiniciarClave($mailer, $usuario, $usuarioToken);
                    $this->addFlash('success', 'Ha sido enviado un mensaje al Correo con las intrucciones a seguir');
                    return $this->redirectToRoute('login');
                }
                else {
                    $this->addFlash('danger', 'Los datos de usuario son incorrecto o no existe');
                }
            }
        }

        return $this->render('security/reiniciar.html.twig', [
            'form' => $form->createView()
        ]);
    }

    private function generarTokenReiniciar(Usuario $usuario) {

        $fchInt = (((int)date('y'))+((int)date('m'))+((int)date('d'))+((int)date('G'))+((int)date('i'))+((int)date('s'))+((int)date('W')))*3;
        $token = (new UriSafeTokenGenerator($fchInt))->generateToken();

        $usuarioToken = new UsuarioToken();
        $usuarioToken->setUsuario($usuario)->setToken($token)
            ->setEstatus(UsuarioToken::ESTATUS_ACTIVO);
        $objDB = $this->getDoctrine()->getManager();
        $objDB->persist($usuarioToken);
        $objDB->flush();
        return $usuarioToken;
    }

    public function reestablecer(Request $request, UserPasswordEncoderInterface $encoder, $token) {

        if($this->isGranted(['IS_AUTHENTICATED_REMEMBERED'])) {
            return $this->redirectToRoute('logout');
        }
        $usuarioToken = $this->getDoctrine()->getRepository(UsuarioToken::class)
                    ->findOneBy(['token' => $token]);

        if($this->validarTokenReiniciar($usuarioToken)) {
            $usuario = $usuarioToken->getUsuario();
            $form = $this->createForm(UsuarioReestablecerType::class, $usuario);
            $form->handleRequest($request);

            if ($form->isSubmitted()) {
                if($form->isValid()) {
                    $objDB = $this->getDoctrine()->getManager();
                    $objDB->getConnection()->beginTransaction();
                    try {
                        $usuario->setPassword($encoder->encodePassword($usuario, $usuario->getPlainPassword()));
                        $usuarioToken->setEstatus(UsuarioToken::ESTATUS_USADO);
                        $objDB->persist($usuario);
                        $objDB->persist($usuarioToken);
                        $objDB->flush();
                        $objDB->getConnection()->commit();
                        $this->addFlash('success', 'La contraseña ha sido reestablecida con exito');
                        return $this->redirectToRoute('login');
                    }
                    catch (Exception $e) {
                        $objDB->getConnection()->rollBack();
                        $this->addFlash('danger', 'Ha surgido un error inesperado, intente de nuevo');
                    }
                }
            }

            return $this->render('security/reestablecer.html.twig', 
                [
                    'form' => $form->createView(),
                    'token' => $token,
                    'usuarioToken' => $usuarioToken, 
                ]
            );
        }
        else {
            $this->addFlash('danger', 'El Link para reestablecer la contraseña es invalido, fue usado o ya expiro');
            return $this->redirectToRoute('login');
        }
    }

    private function validarTokenReiniciar(UsuarioToken $usuarioToken) {
        $valid = false;
        if($usuarioToken) {
            if($usuarioToken->getEstatus() == UsuarioToken::ESTATUS_ACTIVO) {
                $inicio = $usuarioToken->getCreatedAt();
                $actual = new \DateTime();
                $diferencia = $actual->format('U') - $inicio->format('U');
                if($diferencia <= 3600) {
                    $valid = true;
                }
                else {
                    $objDB = $this->getDoctrine()->getManager();
                    $usuarioToken->setEstatus(UsuarioToken::ESTATUS_EXPIRO);
                    $objDB->persist($usuarioToken);
                    $objDB->flush();
                }
            }
        }
        return $valid;
    }

    private function correoReiniciarClave(\Swift_Mailer $mailer, Usuario $usuario, UsuarioToken $usuarioToken) {

        $message = (new \Swift_Message('Reinicio de contraseña de Usuario'))
            ->setFrom('no.responder@netnexo.cl')
            ->setTo($usuario->getEmail());

        $message->setBody(
                $this->renderView(
                    'correo/reiniciar_clave.html.twig', [
                        'usuario' => $usuario,
                        'usuarioToken' => $usuarioToken,
                    ]
                ),
                'text/html'
            );

        $mailer->send($message);
    }


    public function registrar(Request $request, UserPasswordEncoderInterface $encoder)
    {
        if($this->isGranted(['IS_AUTHENTICATED_REMEMBERED'])) {
            return $this->redirectToRoute('logout');
        }

        $usuario = new Usuario();
        $form = $this->createForm(UsuarioNuevoType::class, $usuario);
        $form->handleRequest($request);

        if ($form->isSubmitted()) {
            if($form->isValid()) {
                $objDB = $this->getDoctrine()->getManager();
                $objDB->getConnection()->beginTransaction();
                try {
                    $usuario->setPassword($encoder->encodePassword($usuario, $usuario->getPlainPassword()));

                    $permiso = $this->getDoctrine()->getRepository(Permiso::class)
                        ->findOneBy(['nombre' => 'ROLE_USER']);
                    $usuarioPermiso = new UsuarioPermiso();
                    $usuarioPermiso->setPermiso($permiso);
                    $usuario->addRoles($usuarioPermiso);

                    $objDB->persist($usuario);
                    $objDB->flush();
                    $objDB->getConnection()->commit();
                    $this->addFlash('success', 'Su cuenta ha sido creada con exito');
                    return $this->redirectToRoute('login');
                }
                catch (Exception $e) {
                    $objDB->getConnection()->rollBack();
                    $this->addFlash('danger', 'Ha surgido un error inesperado, intente de nuevo');
                }
            }
            else {
                $this->addFlash('warning', 'Debe ingresar los datos del formulario correctamente');
            }
        }

        return $this->render('security/registrar.html.twig', [
            'form' => $form->createView()
        ]);
    }

    public function login(AuthenticationUtils $authenticationUtils)
    {
        if($this->isGranted('IS_AUTHENTICATED_REMEMBERED')) {
            return $this->redirectToRoute('website_index');
        }

        $error = $authenticationUtils->getLastAuthenticationError();
        $lastUsername = $authenticationUtils->getLastUsername();

        if($error != null) {
            $mensaje = strtr($error->getMessageKey(), $error->getMessageData());
            $this->addFlash('danger', $mensaje);
        }

        return $this->render('security/login.html.twig', 
            ['lastUsername' => $lastUsername]
        );
    }
}