<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

//use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\ArrayCollection;

use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;

/**
 * @ORM\Entity(repositoryClass="App\Repository\OrdenCompraRepository")
 * @ORM\Table(name="orden_compra")
 * @UniqueEntity("correlativo", message ="El correlativo ha sido usado")
 * @ORM\HasLifecycleCallbacks
 */
class OrdenCompra
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_PROCESADA = 1;
    public const ESTATUS_PENDIENTE = 2;
    public const ESTATUS_ANULADA = 3;
    public const ESTATUS_PAGADA = 4;

    public const PAGO_DEPOSITO = 1;
    public const PAGO_TRANSFERENCIA = 2;
    public const PAGO_WEBPAY = 3;

    public const PAGO_LABEL = [
        self::PAGO_DEPOSITO => 'Deposito',
        self::PAGO_TRANSFERENCIA => 'Transferencia',
        self::PAGO_WEBPAY => 'WebPay'
    ];

    public const ESTATUS_LABEL = [
        self::ESTATUS_PROCESADA => 'Procesada',
        self::ESTATUS_PENDIENTE => 'Pendiente',
        self::ESTATUS_ANULADA => 'Anulada',
        self::ESTATUS_PAGADA => 'Pagada'
    ];

    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @ORM\ManyToOne(targetEntity="Usuario")
     * @ORM\JoinColumn(name="usuario_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid
     */
    private $usuario;

    /**
     * @ORM\ManyToOne(targetEntity="Usuario")
     * @ORM\JoinColumn(name="usuario_admin_id", referencedColumnName="id", nullable=true)
     * @Assert\Valid
     */
    private $usuarioAdmin;

    /**
     * @ORM\Column(type="string", length=80, unique=true)
     * @Assert\NotBlank()
     */
    private $correlativo;


    /**
     * @ORM\OneToOne(targetEntity="OrdenFactura", cascade={"persist"})
     * @ORM\JoinColumn(name="orden_factura_id", referencedColumnName="id", nullable=true)
     * @Assert\Valid
     */
    private $ordenFactura;


    /**
     * @var OrdenHosting[]|Collection
     * @ORM\OneToMany(targetEntity="OrdenHosting", mappedBy="ordenCompra", cascade={"persist"})
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid
     */
    private $ordenHosting;


    /**
     * @var OrdenDominio[]|Collection
     * @ORM\OneToMany(targetEntity="OrdenDominio", mappedBy="ordenCompra", cascade={"persist"})
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid
     */
    private $ordenDominio;

    /**
     * @var Transferencias[]|Collection
     * @ORM\OneToMany(targetEntity="Transferencia", mappedBy="ordenCompra", cascade={"persist"})
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid
     */
    private $transferencias;

    /**
     * @var Depositos[]|Collection
     * @ORM\OneToMany(targetEntity="Deposito", mappedBy="ordenCompra", cascade={"persist"})
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid
     */
    private $depositos;

    /**
     * @ORM\Column(type="string", length=120, nullable=true)
     * @Assert\Email(groups={"emailCopia"})
     */
    private $emailCopia;

    /**
     * @ORM\Column(type="string", length=200, nullable=true)
     * @Assert\NotBlank(groups={"procesar"})
     * @Assert\Length(
     *      min = 4,
     *      max = 200,
     *      groups={"procesar"}
     * )
     */
    private $observacion;

    /**
     * @ORM\Column(type="string", length=200, nullable=true)
     * @Assert\NotBlank(groups={"procesar"})
     * @Assert\Length(
     *      min = 4,
     *      max = 200,
     *      groups={"procesar"}
     * )
     */
    private $nota;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":2})
     * @Assert\Choice(callback="getAllEstatus")
     */
    private $estatus;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $fechaCierre;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;


    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;


    /////////////////
    // CONSTRUCTOR //
    /////////////////

    public function __construct()
    {
        $this->ordenHosting = new ArrayCollection();
        $this->ordenDominio = new ArrayCollection();
        $this->transferencias = new ArrayCollection();
        $this->depositos = new ArrayCollection();
        $this->estatus = self::ESTATUS_PENDIENTE;
    }


    /////////////////////
    // GETTER - SETTER //
    /////////////////////

    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }


    /**
     * @param int $id
     *
     * @return self
     */
    public function setId(int $id) : self
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @param Usuario $usuario
     *
     * @return self
     */
    public function setUsuario(Usuario $usuario) : self
    {
        $this->usuario = $usuario;
        return $this;
    }

    /**
     * @return Usuario
     */
    public function getUsuario() : ?Usuario
    {
        return $this->usuario;
    }

    /**
     * @param Usuario $usuario
     *
     * @return self
     */
    public function setUsuarioAdmin(Usuario $usuario) : self
    {
        $this->usuarioAdmin = $usuario;
        return $this;
    }

    /**
     * @return Usuario
     */
    public function getUsuarioAdmin() : ?Usuario
    {
        return $this->usuarioAdmin;
    }


    /**
     * @return string
     */
    public function getCorrelativo() : ?string
    {
        return $this->correlativo;
    }

    /**
     * @param string $correlativo
     *
     * @return self
     */
    public function setCorrelativo(string $correlativo)
    {
        $this->correlativo = $correlativo;
        return $this;
    }

    /**
     * @return OrdenFactura
     */
    public function getOrdenFactura() : ?OrdenFactura
    {
        return $this->ordenFactura;
    }

    /**
     * @param OrdenFactura $ordenFactura
     *
     * @return self
     */
    public function setOrdenFactura(?OrdenFactura $ordenFactura)
    {
        $this->ordenFactura = $ordenFactura;
        return $this;
    }

    /**
     * @return OrdenHosting[]|ArrayCollection
     */
    public function getOrdenHosting()
    {
        return $this->ordenHosting;
    }

    /**
     * @param OrdenHosting $ordenHosting
     *
     * @return self
     */
    public function addOrdenHosting(OrdenHosting $ordenHosting) : self
    {
        if (!$this->ordenHosting->contains($ordenHosting)) {
            $this->ordenHosting->add($ordenHosting);
            $ordenHosting->setOrdenCompra($this);
        }
        return $this;
    }

    /**
     * @param OrdenHosting $ordenHosting
     *
     * @return self
     */
    public function removeOrdenHosting(OrdenHosting $ordenHosting) : self
    {
        $this->ordenHosting->removeElement($ordenHosting);
        return $this;
    }

    /**
     * @return OrdenDominio[]|ArrayCollection
     */
    public function getOrdenDominio()
    {
        return $this->ordenDominio;
    }

    /**
     * @param OrdenDominio $ordenDominio
     *
     * @return self
     */
    public function addOrdenDominio(OrdenDominio $ordenDominio) : self
    {
        if (!$this->ordenDominio->contains($ordenDominio)) {
            $this->ordenDominio->add($ordenDominio);
            $ordenDominio->setOrdenCompra($this);
        }
        return $this;
    }

    /**
     * @param OrdenDominio $ordenDominio
     *
     * @return self
     */
    public function removeOrdenDominio(OrdenDominio $ordenDominio) : self
    {
        $this->ordenDominio->removeElement($ordenDominio);
        return $this;
    }

    /**
     * @return Transferencia[]|ArrayCollection
     */
    public function getTransferencias()
    {
        return $this->transferencias;
    }

    /**
     * @param Transferencia $transferencia
     *
     * @return self
     */
    public function addTransferencia(Transferencia $transf) : self
    {
        if (!$this->transferencias->contains($transf)) {
            $this->transferencias->add($transf);
            $transf->setOrdenCompra($this);
        }
        return $this;
    }

    /**
     * @param Transferencias $transf
     *
     * @return self
     */
    public function removeTransferencia(Transferencia $transf) : self
    {
        $this->transferencias->removeElement($transf);
        return $this;
    }


    /**
     * @return Deposito[]|ArrayCollection
     */
    public function getDepositos()
    {
        return $this->depositos;
    }

    /**
     * @param Deposito $transferencia
     *
     * @return self
     */
    public function addDeposito(Deposito $deposito) : self
    {
        if (!$this->depositos->contains($deposito)) {
            $this->depositos->add($deposito);
            $deposito->setOrdenCompra($this);
        }
        return $this;
    }

    /**
     * @param Deposito $deposito
     *
     * @return self
     */
    public function removeDeposito(Deposito $deposito) : self
    {
        $this->depositos->removeElement($deposito);
        return $this;
    }


    /**
     * @return string
     */
    public function getEmailCopia() : ?string
    {
        return $this->emailCopia;
    }

    /**
     * @param string $email
     *
     * @return self
     */
    public function setEmailCopia(string $email) : self
    {
        $this->emailCopia = $email;
        return $this;
    }

    /**
     * @return string
     */
    public function getObservacion() : ?string
    {
        return $this->observacion;
    }

    /**
     * @param string $observacion
     *
     * @return self
     */
    public function setObservacion($observacion) : self
    {
        $this->observacion = $observacion;
        return $this;
    }

    /**
     * @return string
     */
    public function getNota() : ?string
    {
        return $this->nota;
    }

    /**
     * @param string $nota
     *
     * @return self
     */
    public function setNota($nota) : self
    {
        $this->nota = $nota;
        return $this;
    }


    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param string $estatus
     *
     * @return self
     */
    public function setEstatus($estatus) : self
    {
        $this->estatus = (int)$estatus;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getFechaCierre() : ?\DateTime
    {
        return $this->fechaCierre;
    }

    /**
     * @param string $fechaCierre
     *
     * @return self
     */
    public function setFechaCierre(\DateTime $fechaCierre) : self
    {
        $this->fechaCierre = $fechaCierre;
        return $this;
    }


    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist()
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate()
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }

    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

    /**
     * @return Array
     */
    public function getAllEstatus() {
        return [
            self::ESTATUS_PROCESADA,
            self::ESTATUS_PENDIENTE,
            self::ESTATUS_ANULADA
        ];
    }

}