<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\ArrayCollection;

use App\Validator\Constraints as CustomAssert;

/**
 * @ORM\Entity()
 * @ORM\Table(name="orden_dominio")
 * @ORM\HasLifecycleCallbacks
 * @CustomAssert\DominioDisponible(groups={"dominioDisponible"})
 */
class OrdenDominio
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_PENDIENTE = 2;
    public const ESTATUS_APROBADA = 1;
    public const ESTATUS_RECHAZADA = 3;

    public const ESTATUS_LABEL = [
        self::ESTATUS_PENDIENTE => 'Pendiente',
        self::ESTATUS_APROBADA => 'Aprobada',
        self::ESTATUS_RECHAZADA => 'Rechazada'
    ];


    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;


    /**
     * @ORM\ManyToOne(targetEntity="OrdenCompra", inversedBy="ordenDominio")
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     */

    private $ordenCompra;

    /**
     * @ORM\Column(type="string", length=80)
     * @Assert\NotBlank(groups={"registrarOrd"})
     * @Assert\Length(
     *      min = 3,
     *      max = 80,
     *      groups={"registrarOrd"}
     * )
     * @Assert\Regex(
     *     pattern = "/^([a-z\d](-*[a-z\d])*)(([a-z\d](-*[a-z\d])*))*$/",
     *     message="Ingresar solo el nombre de dominio en minusculas (sin extension ni subdominio)",
     *     groups={"registrarOrd"}
     * )
     */
    
    
    private $nombre;

    /**
     * @ORM\ManyToOne(targetEntity="Extension")
     * @ORM\JoinColumn(name="extension_id", referencedColumnName="id", nullable=false)
     * @Assert\NotBlank(groups={"registrarOrd"})
     */
    private $extension;

    /**
     * @ORM\ManyToOne(targetEntity="Dominio", cascade={"persist"})
     * @ORM\JoinColumn(name="dominio_id", referencedColumnName="id", nullable=true)
     */

    private $dominio;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank(groups={"registrarOrd"})
     * @Assert\Type("integer", groups={"registrarOrd"})
     * @Assert\Range(
     *      min = 1,
     *      max = ExtensionCosto::MAX_ANIO,
     *      groups={"registrarOrd"}
     * )
     */
    private $duracion;

    /**
     * @ORM\Column(type="float")
     * @Assert\NotBlank(groups={"Costo"})
     * @Assert\Regex(
     *     pattern = "/^(([0-9]{1,8})(\.[0-9]{1,2})?)$/",
     *     message="Debe ingresar un numero valido, opcionalmente con 2 decimales. Ejm: 15900.50",
     *     groups={"Costo"}
     * )
     */
    private $costo;


    /**
     * @var \DateTime
     * 
     * @ORM\Column(type="datetime", nullable=true)
     * @Assert\DateTime()
     */
    private $fechaInicio;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime", nullable=true)
     * @Assert\DateTime()
     */
    private $fechaFin;

    /**
     * @ORM\Column(type="string", length=200, nullable=true)
     * @Assert\Length(
     *      min = 4,
     *      max = 200,
     *      groups={"procesarDom"}
     * )
     */
    private $observacion;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":2})
     * @Assert\Choice(callback="getAllEstatus", groups={"procesarDom"})
     */
    private $estatus;


    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;


    /////////////////////
    // GETTER - SETTER //
    /////////////////////


    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @param string $id
     */
    public function setId(int $id)
    {
        $this->id = $id;
    }


    /**
     * @return OrdenCompra
     */
    public function getOrdenCompra() : ?OrdenCompra
    {
        return $this->ordenCompra;
    }

    /**
     * @param OrdenCompra $ordenCompra
     *
     * @return self
     */
    public function setOrdenCompra(OrdenCompra $ordenCompra)
    {
        $this->ordenCompra = $ordenCompra;
        return $this;
    }

    /**
     * @return Extension
     */
    public function getExtension() : ?Extension
    {
        return $this->extension;
    }

    /**
     * @param Extension $extension
     *
     * @return self
     */
    public function setExtension(Extension $extension) : self
    {
        $this->extension = $extension;
        return $this;
    }

    /**
     * @return string
     */
    public function getNombre() : ?string
    {
        return $this->nombre;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setNombre($nombre) : self
    {
        $this->nombre = $nombre;
        return $this;
    }


    /**
     * @return string
     */
    public function getDominio() : ?Dominio
    {
        return $this->dominio;
    }

    /**
     * @param Dominio $dominio
     *
     * @return self
     */
    public function setDominio(Dominio $dominio)
    {
        $this->dominio = $dominio;
        return $this;
    }

    /**
     * @return integer
     */
    public function getDuracion() : ?int
    {
        return $this->duracion;
    }

    /**
     * @param integer $duracion
     *
     * @return self
     */
    public function setDuracion(int $duracion)
    {
        $this->duracion = $duracion;
        return $this;
    }

    /**
     * @return float
     */
    public function getCosto() : ?float
    {
        return $this->costo;
    }

    /**
     * @param float $costo
     *
     * @return self
     */
    public function setCosto($costo)
    {
        $this->costo = $costo;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getFechaInicio() : ?\DateTime
    {
        return $this->fechaInicio;
    }

    /**
     * @param $fechaInicio
     * 
     * @return self
     */
    public function setFechaInicio(\DateTime $fechaInicio) : self
    {
        $this->fechaInicio = $fechaInicio;
        return $this;
    }


    /**
     * @return \DateTime
     */
    public function getFechaFin() : ?\DateTime
    {
        return $this->fechaFin;
    }

    /**
     * @param $fechaFin
     * 
     * @return self
     */
    public function setFechaFin(\DateTime $fechaFin) : self
    {
        $this->fechaFin = $fechaFin;
        return $this;
    }

    /**
     * @return string
     */
    public function getObservacion() : ?string
    {
        return $this->observacion;
    }

    /**
     * @param string $observacion
     *
     * @return self
     */
    public function setObservacion($observacion) : self
    {
        $this->observacion = $observacion;
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param string $estatus
     *
     * @return self
     */
    public function setEstatus($estatus) : self
    {
        $this->estatus = (int)$estatus;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }


    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }


    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

    /**
     * @return Array
     */
    public function getAllEstatus() {
        return [
            self::ESTATUS_PENDIENTE,
            self::ESTATUS_APROBADA,
            self::ESTATUS_RECHAZADA
        ];
    }
}