<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;

use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\ArrayCollection;

/**
 * @ORM\Entity()
 * @ORM\Table(name="plan")
 * @ORM\Entity(repositoryClass="App\Repository\PlanRepository")
 * @ORM\HasLifecycleCallbacks
 * 
 * @UniqueEntity(
 *     "nombre", 
 *     message ="Este nombre de esta Plan ya existe", 
 *     groups={"nuevo", "editar"}
 * )
 */
class Plan
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_ACTIVO = 1;
    public const ESTATUS_INACTIVO = 2;

    public const ESTATUS_LABEL = [
        self::ESTATUS_ACTIVO => 'Activo',
        self::ESTATUS_INACTIVO => 'Inactivo'
    ];

    public const MAX_ANIO = 5;

    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=50, unique=true)
     * @Assert\NotBlank(groups={"nuevo", "editar"})
     * @Assert\Length(
     *      min = 3,
     *      max = 50,
     *      groups={"nuevo", "editar"}
     * )
     */
    private $nombre;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=500)
     * @Assert\NotBlank(groups={"nuevo", "editar"})
     * @Assert\Length(
     *      min = 3,
     *      max = 500,
     *      groups={"nuevo", "editar"}
     * )
     */
    private $descripcion;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":1})
     */
    private $nivel;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":1})
     * @Assert\Choice(
     *     {App\Entity\Plan::ESTATUS_ACTIVO},
     *     groups={"DestinoPlan"},
     * )
     */
    private $estatus;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;

    /**
     * @ORM\OneToMany(targetEntity="PlanDetalle", mappedBy="plan",
     *     cascade={"persist"})
     * @ORM\OrderBy({"caracteristica" = "DESC"})
     */
    private $detalles;

    /**
     * @ORM\OneToMany(targetEntity="PlanCosto", mappedBy="plan",
     *     cascade={"persist"})
     */
    private $costos;


    /////////////////
    // CONSTRUCTOR //
    /////////////////

    public function __construct()
    {
        $this->detalles = new ArrayCollection();
        $this->costos = new ArrayCollection();
        $this->nivel = 1;
        $this->estatus = self::ESTATUS_ACTIVO;
    }

    /////////////////////
    // GETTER - SETTER //
    /////////////////////

    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getNombre() : ?string
    {
        return $this->nombre;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setNombre($nombre) : self
    {
        $this->nombre = $nombre;
        return $this;
    }

    /**
     * @return string
     */
    public function getDescripcion() : ?string
    {
        return $this->descripcion;
    }

    /**
     * @param string $descripcion
     *
     * @return self
     */
    public function setDescripcion($descripcion) : self
    {
        $this->descripcion = $descripcion;
        return $this;
    }

    /**
     * @return int
     */
    public function getNivel() : ?int
    {
        return $this->nivel;
    }

    /**
     * @param string $nivel
     *
     * @return self
     */
    public function setNivel($nivel) : self
    {
        $this->nivel = $nivel;
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param int $estatus
     *
     * @return self
     */
    public function setEstatus($estatus) : self
    {
        $this->estatus = $estatus;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /**
     * @return PlanDetalle[]|ArrayCollection
     */
    public function getDetalles()
    {
        return $this->detalles;
    }

    /**
     * @param PlanDetalle $planDetalle
     *
     * @return self
     */
    public function addDetalle(PlanDetalle $planDetalle) : self
    {
        if (!$this->detalles->contains($planDetalle)) {
            $this->detalles->add($planDetalle);
            $planDetalle->setPlan($this);
        }

        return $this;
    }

    /**
     * @param PlanDetalle $planDetalle
     *
     * @return self
     */
    public function removeDetalle(PlanDetalle $planDetalle) : self
    {
        $this->detalles->removeElement($planDetalle);

        return $this;
    }

    /**
     * @return PlanCosto[]|ArrayCollection
     */
    public function getCostos()
    {
        return $this->costos;
    }

    /**
     * @param PlanCosto $planCosto
     *
     * @return self
     */
    public function addCosto(PlanCosto $planCosto) : self
    {
        if (!$this->costos->contains($planCosto)) {
            $this->costos->add($planCosto);
            $planCosto->setPlan($this);
        }
        return $this;
    }

    /**
     * @param PlanCosto $planCosto
     *
     * @return self
     */
    public function removeCosto(PlanCosto $planCosto) : self
    {
        $this->costos->removeElement($planCosto);
        return $this;
    }

    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist()
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate()
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }

    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

}