<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

use Symfony\Component\HttpFoundation\File\File;
use Vich\UploaderBundle\Mapping\Annotation as Vich;

/**
 * @ORM\Entity()
 * @ORM\Table(name="transferencia")
 * @ORM\HasLifecycleCallbacks
 * @Vich\Uploadable
 */
class Transferencia
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_ACTIVO = 1;
    public const ESTATUS_INACTIVO = 2;

    public const ESTATUS = [
        self::ESTATUS_ACTIVO,
        self::ESTATUS_INACTIVO
    ];


    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @Assert\NotBlank(groups={"transferenciaNuevo"})
     * @Assert\File(
     *     maxSize="2M",
     *     mimeTypes={"image/png", "image/jpeg"},
     *     groups={"transferenciaNuevo"}
     * )
     * @Vich\UploadableField(mapping="pagos", fileNameProperty="imageName")
     * 
     * @var File
     */
    private $imageFile;

    /**
     * @ORM\Column(type="string", length=255)
     *
     * @var string
     */
    private $imageName;

    /**
     * @ORM\ManyToOne(targetEntity="Banco")
     * @ORM\JoinColumn(name="banco_id", referencedColumnName="id", nullable=false)
     * @Assert\NotBlank(groups={"transferenciaNuevo"})
     */
    private $banco;

    /**
     * @ORM\ManyToOne(targetEntity="Cuenta")
     * @ORM\JoinColumn(name="cuenta_id", referencedColumnName="id", nullable=false)
     * @Assert\NotBlank(groups={"transferenciaNuevo"})
     */
    private $cuenta;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     * @Assert\NotBlank(groups={"transferenciaNuevo"})
     * @Assert\Date(groups={"transferenciaNuevo"})
     * @Assert\LessThanOrEqual(
     *      "today", 
     *      message = "La fecha no puede ser mayor a la actual",     
     *      groups={"transferenciaNuevo"}
     * )
     */
    private $fecha;

    /**
     * @ORM\Column(type="string", length=200, nullable=true)
     * @Assert\Length(
     *      min = 4,
     *      max = 200,
     *      groups={"transferenciaNuevo"}
     * )
     */
    private $observacion;

    /**
     * @ORM\ManyToOne(targetEntity="OrdenCompra", inversedBy="transferencias")
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     */
    private $ordenCompra;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":1})
     * @Assert\Choice(callback="getAllEstatus", groups={"transferenciaNuevo"})
     */
    private $estatus;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;


    /////////////////
    // CONSTRUCTOR //
    /////////////////
    public function __construct()
    {
        $this->estatus = 1;
    }


    /////////////////////
    // GETTER - SETTER //
    /////////////////////

    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * If manually uploading a file (i.e. not using Symfony Form) ensure an instance
     * of 'UploadedFile' is injected into this setter to trigger the update. If this
     * bundle's configuration parameter 'inject_on_load' is set to 'true' this setter
     * must be able to accept an instance of 'File' as the bundle will inject one here
     * during Doctrine hydration.
     *
     * @param File|\Symfony\Component\HttpFoundation\File\UploadedFile $imageFile
     */
    public function setImageFile(?File $imageFile = null): void
    {
        $this->imageFile = $imageFile;

        if (null !== $imageFile) {
            // It is required that at least one field changes if you are using doctrine
            // otherwise the event listeners won't be called and the file is lost
            $this->updatedAt = new \DateTimeImmutable();
        }
    }

    public function getImageFile(): ?File
    {
        return $this->imageFile;
    }

    public function setImageName($imageName): void
    {
        $this->imageName = $imageName;
    }

    public function getImageName(): ?string
    {
        return $this->imageName;
    }

    /**
     * @return Banco
     */
    public function getBanco() : ?Banco
    {
        return $this->banco;
    }

    /**
     * @param Banco $banco
     *
     * @return self
     */
    public function setBanco(Banco $banco)
    {
        $this->banco = $banco;
        return $this;
    }

    /**
     * @return Cuenta
     */
    public function getCuenta() : ?Cuenta
    {
        return $this->cuenta;
    }

    /**
     * @param Cuenta $cuenta
     *
     * @return self
     */
    public function setCuenta(Cuenta $cuenta)
    {
        $this->cuenta = $cuenta;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getFecha() : ?\DateTime
    {
        return $this->fecha;
    }

    /**
     * @param \DateTime $fecha
     * 
     * @return self
     */
    public function setFecha(?\DateTime $fecha) : ?self
    {
        $this->fecha = $fecha;
        return $this;
    }

        /**
     * @return string
     */
    public function getObservacion() : ?string
    {
        return $this->observacion;
    }

    /**
     * @param string $observacion
     *
     * @return self
     */
    public function setObservacion($observacion) : self
    {
        $this->observacion = $observacion;
        return $this;
    }

    /**
     * @return OrdenCompra
     */
    public function getOrdenCompra() : ?OrdenCompra
    {
        return $this->ordenCompra;
    }

    /**
     * @param OrdenCompra $ordenCompra
     *
     * @return self
     */
    public function setOrdenCompra(OrdenCompra $ordenCompra)
    {
        $this->ordenCompra = $ordenCompra;
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setEstatus(int $estatus) : self
    {
        $this->estatus = $estatus;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist()
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate()
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }

    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

    public function getAllEstatus() {
        return self::ESTATUS;
    }

}