<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

use Symfony\Component\HttpFoundation\File\File;
use Vich\UploaderBundle\Mapping\Annotation as Vich;

/**
 * @ORM\Entity()
 * @ORM\Table(name="depositos")
 * @ORM\HasLifecycleCallbacks
 * @Vich\Uploadable
 */
class Deposito
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_ACTIVO = 1;
    public const ESTATUS_INACTIVO = 2;

    public const TIPO_EFECTIVO = 1;
    public const TIPO_CHEQUE = 2;

    public const TIPO_LABEL = [
        self::TIPO_EFECTIVO => 'Efectivo',
        self::TIPO_CHEQUE => 'Cheque',
    ];

    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @Assert\NotBlank(groups={"depositoNuevo"})
     * @Assert\File(
     *     maxSize="2M",
     *     mimeTypes={"image/png", "image/jpeg"},
     *     groups={"depositoNuevo"}
     * )
     * @Vich\UploadableField(mapping="pagos", fileNameProperty="imageName")
     * 
     * @var File
     */
    private $imageFile;

    /**
     * @ORM\Column(type="string", length=255)
     *
     * @var string
     */
    private $imageName;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank(groups={"depositoNuevo"})
     * @Assert\Choice(callback="getAllTipo", groups={"depositoNuevo"})
     */
    private $tipo;

    /**
     * @ORM\ManyToOne(targetEntity="Cuenta")
     * @ORM\JoinColumn(name="cuenta_id", referencedColumnName="id", nullable=false)
     * @Assert\NotBlank(groups={"depositoNuevo"})
     */
    private $cuenta;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime", nullable=false)
     * @Assert\NotBlank(groups={"depositoNuevo"})
     * @Assert\Date(groups={"depositoNuevo"})
     * @Assert\LessThanOrEqual("today", groups={"depositoNuevo"})
     */
    private $fecha;

    /**
     * @ORM\Column(type="string", length=200, nullable=true)
     * @Assert\Length(
     *      min = 4,
     *      max = 200,
     *      groups={"depositoNuevo"}
     * )
     */
    private $observacion;

    /**
     * @ORM\ManyToOne(targetEntity="OrdenCompra", inversedBy="depositos")
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     */
    private $ordenCompra;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":1})
     * @Assert\Choice(callback="getAllEstatus", groups={"depositoNuevo"})
     */
    private $estatus;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;


    /////////////////
    // CONSTRUCTOR //
    /////////////////
    public function __construct()
    {
        $this->estatus = 1;
    }


    /////////////////////
    // GETTER - SETTER //
    /////////////////////

    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @param File|\Symfony\Component\HttpFoundation\File\UploadedFile $imageFile
     */
    public function setImageFile(?File $imageFile = null): void
    {
        $this->imageFile = $imageFile;

        if (null !== $imageFile) {
            // It is required that at least one field changes if you are using doctrine
            // otherwise the event listeners won't be called and the file is lost
            $this->updatedAt = new \DateTimeImmutable();
        }
    }

    public function getImageFile(): ?File
    {
        return $this->imageFile;
    }

    public function setImageName($imageName): void
    {
        $this->imageName = $imageName;
    }

    public function getImageName(): ?string
    {
        return $this->imageName;
    }

    /**
     * @return int
     */
    public function getTipo() : ?int
    {
        return $this->tipo;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setTipo(int $tipo) : self
    {
        $this->tipo = $tipo;
        return $this;
    }


    /**
     * @return Cuenta
     */
    public function getCuenta() : ?Cuenta
    {
        return $this->cuenta;
    }

    /**
     * @param Cuenta $cuenta
     *
     * @return self
     */
    public function setCuenta(Cuenta $cuenta)
    {
        $this->cuenta = $cuenta;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getFecha() : ?\DateTime
    {
        return $this->fecha;
    }

    /**
     * @param \DateTime $fecha
     * 
     * @return self
     */
    public function setFecha(?\DateTime $fecha) : ?self
    {
        $this->fecha = $fecha;
        return $this;
    }

        /**
     * @return string
     */
    public function getObservacion() : ?string
    {
        return $this->observacion;
    }

    /**
     * @param string $observacion
     *
     * @return self
     */
    public function setObservacion($observacion) : self
    {
        $this->observacion = $observacion;
        return $this;
    }

    /**
     * @return OrdenCompra
     */
    public function getOrdenCompra() : ?OrdenCompra
    {
        return $this->ordenCompra;
    }

    /**
     * @param OrdenCompra $ordenCompra
     *
     * @return self
     */
    public function setOrdenCompra(OrdenCompra $ordenCompra)
    {
        $this->ordenCompra = $ordenCompra;
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setEstatus(int $estatus) : self
    {
        $this->estatus = $estatus;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist()
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate()
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }

    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

    public function getAllEstatus() {
        return [self::ESTATUS_ACTIVO, self::ESTATUS_INACTIVO];
    }

    public function getAllTipo() {
        return [self::TIPO_EFECTIVO, self::TIPO_CHEQUE];
    }

    public function getTipoLabel($label) {
        return self::TIPO_LABEL[$label];
    }

}