<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity()
 * @ORM\Table(name="dns")
 * @ORM\HasLifecycleCallbacks
 */
class Dns
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_ACTIVO = 1;
    public const ESTATUS_INACTIVO = 2;

    public const ESTATUS_LABEL = [
        self::ESTATUS_ACTIVO => 'Activo',
        self::ESTATUS_INACTIVO => 'Inactivo'
    ];

    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @ORM\Column(type="string", length=100, nullable=true)
     * @Assert\NotBlank(groups={"nuevoDns"})
     * @Assert\Length(
     *      min = 4,
     *      max = 100,
     *      groups={"nuevoDns"}
     * )
     * @Assert\Regex(
     *     pattern = "/^(?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?\.)+[a-z0-9][a-z0-9-]{0,61}[a-z0-9]$/",
     *     message="Debe ingresar un nombre de Dominio valido",
     *     groups={"nuevoDns"}
     * )
     */
    private $dominio;

    /**
     * @ORM\Column(type="string", length=100, nullable=true)
     * @Assert\NotBlank(groups={"nuevoDns"})
     * @Assert\Length(
     *      min = 4,
     *      max = 100,
     *      groups={"nuevoDns"}
     * )
     * @Assert\Regex(
     *     pattern = "/^((25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.){3}(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)(:[1-9][0-9]*)?$/",
     *     message="Debe ingrese una IP valida",
     *     groups={"nuevoDns"}
     * )
     */
    private $ip;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default": 1}, nullable=false)
     */
    private $estatus;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $updatedAt;


    ////////////////
    // CONTRUCTOR //
    ////////////////

    public function __construct()
    {
        $this->estatus = self::ESTATUS_ACTIVO;
    }

    /////////////////////
    // GETTER - SETTER //
    /////////////////////


    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @param string $id
     */
    public function setId(int $id)
    {
        $this->id = $id;
    }

    /**
     * @return string
     */
    public function getDominio() : ?string
    {
        return $this->dominio;
    }

    /**
     * @param string $dominio
     *
     * @return self
     */
    public function setDominio(string $dominio)
    {
        $this->dominio = $dominio;
        return $this;
    }

    /**
     * @return string
     */
    public function getIp() : ?string
    {
        return $this->ip;
    }

    /**
     * @param string $ip
     *
     * @return self
     */
    public function setIp(string $ip)
    {
        $this->ip = $ip;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param int $estatus
     *
     * @return self
     */
    public function setEstatus(int $estatus) : self
    {
        $this->estatus = $estatus;
        return $this;
    }


    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }


}