<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity()
 * @ORM\Entity(repositoryClass="App\Repository\DominioRepository")
 * @ORM\Table(name="dominio")
 * @ORM\HasLifecycleCallbacks
 */
class Dominio
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_ACTIVO = 1;
    public const ESTATUS_INACTIVO = 2;

    public const ESTATUS = [
        self::ESTATUS_ACTIVO,
        self::ESTATUS_INACTIVO
    ];

    public const ESTATUS_LABEL = [
        self::ESTATUS_ACTIVO => 'Activo',
        self::ESTATUS_INACTIVO => 'Inactivo'
    ];

    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @ORM\ManyToOne(targetEntity="Usuario")
     * @ORM\JoinColumn(name="usuario_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid
     */
    private $usuario;

    /**
     * @ORM\Column(type="string", length=80)
     * @Assert\NotBlank()
     * @Assert\Length(
     *      min = 1,
     *      max = 80
     * )
     */
    private $nombre;

    /**
     * @ORM\ManyToOne(targetEntity="Extension")
     * @ORM\JoinColumn(name="extension_id", referencedColumnName="id", nullable=false)
     * @Assert\NotBlank()
     * @Assert\Valid
     */
    private $extension;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":1})
     * @Assert\NotBlank()
     * @Assert\Choice(callback="getAllEstatus")
     */
    private $estatus;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;


    /////////////////////
    // GETTER - SETTER //
    /////////////////////

    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @return Usuario
     */
    public function getUsuario() : ?Usuario
    {
        return $this->usuario;
    }

    /**
     * @param Usuario $usuario
     *
     * @return self
     */
    public function setUsuario(Usuario $usuario) : self
    {
        $this->usuario = $usuario;
        return $this;
    }

    /**
     * @return Extension
     */
    public function getExtension() : ?Extension
    {
        return $this->extension;
    }

    /**
     * @param Extension $extension
     *
     * @return self
     */
    public function setExtension(Extension $extension) : self
    {
        $this->extension = $extension;
        return $this;
    }

    /**
     * @return string
     */
    public function getNombre() : ?string
    {
        return $this->nombre;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setNombre(string $nombre) : self
    {
        $this->nombre = $nombre;
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setEstatus(int $estatus) : self
    {
        $this->estatus = $estatus;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist()
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate()
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }

    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

    public function getAllEstatus() {
        return self::ESTATUS;
    }
}