<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\ArrayCollection;

use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity(repositoryClass="App\Repository\ExtensionRepository")
 * @ORM\Table(name="extension")
 * @ORM\HasLifecycleCallbacks
 */
class Extension
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_ACTIVO = 1;
    public const ESTATUS_INACTIVO = 2;

    public const ESTATUS_LABEL = [
        self::ESTATUS_ACTIVO => 'Activo',
        self::ESTATUS_INACTIVO => 'Inactivo'
    ];

    public const MAX_ANIO = 5;


    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @var int
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @var string
     *
     * @ORM\Column(type="string", length=10, unique=true)
     * @Assert\NotBlank()
     * @Assert\Length(
     *      min = 1,
     *      max = 10
     * )
     */
    private $nombre;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;

    /**
     * @ORM\OneToMany(targetEntity="ExtensionCosto", mappedBy="extension", cascade={"persist"})
     * @ORM\JoinColumn(name="extension_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid
     */
    private $costo;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":1})
     * @Assert\Choice({App\Entity\Extension::ESTATUS_ACTIVO},
     *     groups={"RegistroDom"})
     */
    private $estatus;


    /////////////////
    // CONSTRUCTOR //
    /////////////////

    public function __construct()
    {
        $this->costo = new ArrayCollection();
        $this->estatus = 1;
    }

    /////////////////////
    // GETTER - SETTER //
    /////////////////////

    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @return string
     */
    public function getNombre() : ?string
    {
        return $this->nombre;
    }

    /**
     * @param string $nombre
     *
     * @return self
     */
    public function setNombre(string $nombre) : self
    {
        $this->nombre = $nombre;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /**
     * @return ExtensionCosto[]|ArrayCollection
     */
    public function getCosto()
    {
        return $this->costo;
    }

    /**
     * @param ExtensionCosto $costo
     *
     * @return self
     */
    public function addCosto(ExtensionCosto $costo) : self
    {
        if (!$this->costo->contains($costo)) {
            $this->costo->add($costo);
            $costo->setExtension($this);
        }
        return $this;
    }

    /**
     * @param ExtensionCosto $costo
     *
     * @return self
     */
    public function removeCosto(ExtensionCosto $costo) : self
    {
        $this->costo->removeElement($costo);
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param int $estatus
     *
     * @return self
     */
    public function setEstatus(int $estatus) : self
    {
        $this->estatus = $estatus;
        return $this;
    }

    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist()
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate()
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }

    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

}