<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;

use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\ArrayCollection;

use Symfony\Component\Validator\Constraints as Assert;

/**
 * @ORM\Entity()
 * @ORM\Table(name="orden_hosting")
 * @ORM\HasLifecycleCallbacks
 */
class OrdenHosting
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_PENDIENTE = 2;
    public const ESTATUS_APROBADA = 1;
    public const ESTATUS_RECHAZADA = 3;

    public const ESTATUS_LABEL = [
        self::ESTATUS_PENDIENTE => 'Pendiente',
        self::ESTATUS_APROBADA => 'Aprobada',
        self::ESTATUS_RECHAZADA => 'Rechazada'
    ];

    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @ORM\ManyToOne(targetEntity="OrdenCompra", inversedBy="ordenHosting")
     * @ORM\JoinColumn(name="orden_compra_id", referencedColumnName="id", nullable=false)
     */
    private $ordenCompra;

    /**
     * @ORM\ManyToOne(targetEntity="Hosting", cascade={"persist"})
     * @ORM\JoinColumn(name="hosting_id", referencedColumnName="id", nullable=true)
     */
    private $hosting;


    /**
     * @ORM\ManyToOne(targetEntity="Plan")
     * @ORM\JoinColumn(name="plan_origen_id", referencedColumnName="id", nullable=true)
     */
    private $planOrigen;

    /**
     * @ORM\ManyToOne(targetEntity="Plan")
     * @ORM\JoinColumn(name="plan_destino_id", referencedColumnName="id", nullable=false)
     */
    private $planDestino;

    /**
     * @ORM\Column(type="string", length=100, nullable=true)
     * @Assert\NotBlank(groups={"registrarOrd"})
     * @Assert\Length(
     *      min = 4,
     *      max = 100,
     *      groups={"registrarOrd"}
     * )
     * @Assert\Regex(
     *     pattern = "/^(?:[a-z0-9](?:[a-z0-9-]{0,61}[a-z0-9])?\.)+[a-z0-9][a-z0-9-]{0,61}[a-z0-9]$/",
     *     message="Debe ingresar un nombre de Dominio valido",
     *     groups={"registrarOrd"}
     * )
     */
    private $detalle;

    /**
     * @ORM\Column(type="string", length=200, nullable=true)
     * @Assert\Length(
     *      min = 4,
     *      max = 200,
     *      groups={"procesarHst"}
     * )
     */
    private $observacion;

    /**
     * @ORM\Column(type="integer")
     * @Assert\NotBlank(groups={"registrarOrd"})
     * @Assert\Type("integer", groups={"registrarOrd"})
     * @Assert\Range(
     *      min = 1,
     *      max = Plan::MAX_ANIO,
     *      groups={"registrarOrd"}
     * )
     */
    private $duracion;

    /**
     * @ORM\Column(type="float")
     * @Assert\NotBlank(groups={"Costos"})
     * @Assert\Regex(
     *     pattern = "/^(([0-9]{1,8})(\.[0-9]{1,2})?)$/",
     *     message="Debe ingresar un numero valido, opcionalmente con 2 decimales. Ejm: 15900.50",
     *     groups={"Costos"}
     * )
     * @Assert\Range(
     *      min = 2,
     *      max = Plan::MAX_ANIO,
     *      groups={"Costos"}
     * )
     */
    private $costo;


    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime", nullable=true)
     * @Assert\DateTime()
     */
    private $fechaInicio;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime", nullable=true)
     */
    private $fechaFin;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default":2})
     * @Assert\Choice(callback="getAllEstatus", groups={"procesarHst"})
     */
    private $estatus;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;


    /////////////////////
    // GETTER - SETTER //
    /////////////////////


    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @param string $id
     */
    public function setId(int $id)
    {
        $this->id = $id;
    }


    /**
     * @return OrdenCompra
     */
    public function getOrdenCompra() : ?OrdenCompra
    {
        return $this->ordenCompra;
    }

    /**
     * @param OrdenCompra $ordenCompra
     *
     * @return self
     */
    public function setOrdenCompra(OrdenCompra $ordenCompra)
    {
        $this->ordenCompra = $ordenCompra;
        return $this;
    }

    /**
     * @return string
     */
    public function getHosting() : ?Hosting
    {
        return $this->hosting;
    }

    /**
     * @param Hosting $hosting
     *
     * @return self
     */
    public function setHosting(Hosting $hosting)
    {
        $this->hosting = $hosting;
        return $this;
    }


    /**
     * @return OrdenExpansion[]|ArrayCollection
     */
    /*public function getOrdenExpansion()
    {
        return $this->ordenExpansion;
    }*/

    /**
     * @param OrdenExpansion $ordenExpansion
     *
     * @return self
     */
    /*public function addOrdenExpansion(OrdenExpansion $ordenExpansion) : self
    {
        if (!$this->ordenExpansion->contains($ordenExpansion)) {
            $this->ordenExpansion->add($ordenExpansion);
        }
        return $this;
    }*/

    /**
     * @param OrdenExpansion $ordenExpansion
     *
     * @return self
     */
    /*public function removeOrdenExpansion(OrdenExpansion $ordenExpansion) : self
    {
        $this->ordenExpansion->removeElement($ordenExpansion);
        return $this;
    }*/

    /**
     * @return Plan
     */
    public function getPlanOrigen() : ?Plan
    {
        return $this->planOrigen;
    }

    /**
     * @param Plan $planOrigen
     *
     * @return self
     */
    public function setPlanOrigen(Plan $planOrigen) : self
    {
        $this->planOrigen = $planOrigen;
        return $this;
    }

    /**
     * @return Plan
     */
    public function getPlanDestino() : ?Plan
    {
        return $this->planDestino;
    }

    /**
     * @param Plan $planDestino
     *
     * @return self
     */
    public function setPlanDestino(Plan $planDestino) : self
    {
        $this->planDestino = $planDestino;
        return $this;
    }

    /**
     * @return string
     */
    public function getDetalle() : ?string
    {
        return $this->detalle;
    }

    /**
     * @param string $detalle
     *
     * @return self
     */
    public function setDetalle($detalle) : self
    {
        $this->detalle = $detalle;
        return $this;
    }

    /**
     * @return string
     */
    public function getObservacion() : ?string
    {
        return $this->observacion;
    }

    /**
     * @param string $observacion
     *
     * @return self
     */
    public function setObservacion($observacion) : self
    {
        $this->observacion = $observacion;
        return $this;
    }

    /**
     * @return integer
     */
    public function getDuracion() : ?int
    {
        return $this->duracion;
    }

    /**
     * @param integer $duracion
     *
     * @return self
     */
    public function setDuracion(int $duracion)
    {
        $this->duracion = $duracion;
        return $this;
    }

    /**
     * @return float
     */
    public function getCosto() : ?float
    {
        return $this->costo;
    }

    /**
     * @param float $costo
     *
     * @return self
     */
    public function setCosto(float $costo)
    {
        $this->costo = $costo;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getFechaInicio() : ?\DateTime
    {
        return $this->fechaInicio;
    }

    /**
     * @param $fechaInicio
     *
     * @return self
     */
    public function setFechaInicio(\DateTime $fechaInicio) : self
    {
        $this->fechaInicio = $fechaInicio;
        return $this;
    }


    /**
     * @return \DateTime
     */
    public function getFechaFin() : ?\DateTime
    {
        return $this->fechaFin;
    }

    /**
     * @param $fechaFin
     *
     * @return self
     */
    public function setFechaFin(\DateTime $fechaFin) : self
    {
        $this->fechaFin = $fechaFin;
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param string $estatus
     *
     * @return self
     */
    public function setEstatus($estatus) : self
    {
        $this->estatus = (int)$estatus;
        return $this;
    }

    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }


    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }


    /////////////////////
    // OTRAS FUNCIONES //
    /////////////////////

    /**
     * @return Array
     */
    public function getAllEstatus() {
        return [
            self::ESTATUS_PENDIENTE,
            self::ESTATUS_APROBADA,
            self::ESTATUS_RECHAZADA
        ];
    }
}