<?php

namespace App\Entity;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Bridge\Doctrine\Validator\Constraints\UniqueEntity;

use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\ArrayCollection;

/**
 * @ORM\Entity()
 * @ORM\Table(name="usuario")
 * @ORM\HasLifecycleCallbacks
 * @ORM\Entity(repositoryClass="App\Repository\UsuarioRepository")
 * 
 * @UniqueEntity(
 *     "username", 
 *     message ="Este nombre de usuario ya exite", 
 *     groups={"nuevo"}
 * )
 * @UniqueEntity(
 *     "email", 
 *     message ="Este correo electronico ya esta en uso", 
 *     groups={"nuevo"}
 * )
 */
class Usuario implements UserInterface, \Serializable
{
    ////////////////
    // CONSTANTES //
    ////////////////

    public const ESTATUS_ACTIVO = 1;
    public const ESTATUS_INACTIVO = 2;

    public const ESTATUS_LABEL = [
        self::ESTATUS_ACTIVO => 'Activo',
        self::ESTATUS_INACTIVO => 'Inactivo'
    ];


    ///////////////
    // ATRIBUTOS //
    ///////////////

    /**
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private $id;

    /**
     * @ORM\Column(type="string", length=25, unique=true)
     * @Assert\NotBlank(groups={"nuevo"})
     * @Assert\Type("alpha", groups={"nuevo"})
     * @Assert\Length(
     *      min = 6,
     *      max = 25,
     *      groups={"nuevo"}
     * )
     */
    private $username;

    /**
     * @Assert\NotBlank(groups={"nuevo", "resetpass"})
     * @Assert\Length(
     *      min = 8,
     *      max = 64,
     *      groups={"nuevo", "resetpass"}
     * )
     */
    private $plainPassword;


    /**
     * @ORM\Column(type="string", length=64)
     */
    private $password;

    /**
     * @ORM\Column(type="string", length=191, unique=true)
     * @Assert\NotBlank(groups={"nuevo"})
     * @Assert\Email(groups={"nuevo"})
     */
    private $email;

    /**
     * @var int
     *
     * @ORM\Column(type="integer", options={"default": 1}, nullable=false)
     */
    private $estatus;

    /**
     * @ORM\OneToOne(targetEntity="Persona", cascade={"persist"})
     * @ORM\JoinColumn(name="persona_id", referencedColumnName="id", nullable=false)
     * @Assert\Valid(groups={"nuevoUsuario"})
     */
    private $persona;

    /**
     * @ORM\OneToMany(targetEntity="UsuarioPermiso", mappedBy="usuario", cascade={"persist"})
     * @ORM\JoinColumn(name="usuario_id", referencedColumnName="id", nullable=false)
     */
    private $roles;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $createdAt;

    /**
     * @var \DateTime
     *
     * @ORM\Column(type="datetime")
     */
    private $updatedAt;


    ////////////////
    // CONTRUCTOR //
    ////////////////

    public function __construct()
    {
        $this->estatus = 1;
        $this->roles = new ArrayCollection();
        // may not be needed, see section on salt below
        // $this->salt = md5(uniqid('', true));
    }


    /////////////////////
    // GETTER - SETTER //
    /////////////////////


    /**
     * @return int
     */
    public function getId() : ?int
    {
        return $this->id;
    }

    /**
     * @param string $id
     *
     * @return self
     */
    public function setId(int $id)
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @return string
     */
    public function getUsername() : ?string
    {
        return $this->username;
    }

    /**
     * @param string $username
     *
     * @return self
     */
    public function setUsername(string $username)
    {
        $this->username = strtolower($username);
        return $this;
    }

    /**
     * @return string
     */
    public function getPersona() : ?Persona
    {
        return $this->persona;
    }

    /**
     * @param Persona $persona
     *
     * @return self
     */
    public function setPersona(Persona $persona)
    {
        $this->persona = $persona;
        return $this;
    }

    public function getSalt()
    {
        // you *may* need a real salt depending on your encoder
        // see section on salt below
        return null;
    }

    /**
     * @param string $plainPassword
     *
     * @return self
     */
    public function setPlainPassword(string $plainPassword)
    {
        $this->plainPassword = $plainPassword;
        return $this;
    }

    /**
     * @return string
     */
    public function getPlainPassword()
    {
        return $this->plainPassword;
    }

    /**
     * @param string $password
     *
     * @return self
     */
    public function setPassword(string $password)
    {
        $this->password = $password;
        return $this;
    }

    /**
     * @return string
     */
    public function getPassword()
    {
        return $this->password;
    }

    /**
     * @return string
     */
    public function getEmail() : ?string
    {
        return $this->email;
    }

    /**
     * @param string $email
     *
     * @return self
     */
    public function setEmail(string $email)
    {
        $this->email = strtolower($email);
        return $this;
    }

    /**
     * @return int
     */
    public function getEstatus() : ?int
    {
        return $this->estatus;
    }

    /**
     * @param int $estatus
     *
     * @return self
     */
    public function setEstatus(int $estatus) : self
    {
        $this->estatus = $estatus;
        return $this;
    }


    /**
     * @return array
     */

    public function getRoles()
    {
        $roles = [];
        foreach($this->roles as $usuarioPermiso) {
            $roles[] = $usuarioPermiso->getPermiso()->getNombre();
        }
        return $roles;
    }

    /**
     * @param UsuarioPermiso $roles
     *
     * @return self
     */
    public function addRoles(UsuarioPermiso $roles) : self
    {
        if (!$this->roles->contains($roles)) {
            $roles->setUsuario($this);
            $this->roles->add($roles);
        }
        return $this;
    }

    /**
     * @param UsuarioPermiso $roles
     *
     * @return self
     */
    public function removeDetalle(UsuarioPermiso $roles) : self
    {
        $this->detalles->removeElement($roles);
        return $this;
    }


    /**
     * @return \DateTime
     */
    public function getCreatedAt() : ?\DateTime
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTime
     */
    public function getUpdateAt() : ?\DateTime
    {
        return $this->updatedAt;
    }

    /////////////////////////
    // LIFECYCLE CALLBACKS //
    /////////////////////////

    /**
     * @ORM\PrePersist
     */
    public function prePersist()
    {
        $this->createdAt = new \DateTime();
        $this->updatedAt = new \DateTime();
    }

    /**
     * @ORM\PreUpdate
     */
    public function preUpdate()
    {
        $this->updatedAt = new \DateTime();
    }


    /////////////////////
    // OTRAS FUNCIONEs //
    /////////////////////


    public function eraseCredentials()
    {
    }

    /** @see \Serializable::serialize() */
    public function serialize()
    {
        return serialize(array(
            $this->id,
            $this->username,
            $this->password,
            // see section on salt below
            // $this->salt,
        ));
    }

    /** @see \Serializable::unserialize() */
    public function unserialize($serialized)
    {
        list (
            $this->id,
            $this->username,
            $this->password,
            // see section on salt below
            // $this->salt
        ) = unserialize($serialized, array('allowed_classes' => false));
    }
}